/* ****************************************************************************
 * Copyright: 2017-2025 RAYLASE GmbH
 * This source code is the proprietary confidential property of RAYLASE GmbH.
 * Reproduction, publication, or any form of distribution to
 * any party other than the licensee is strictly prohibited.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#pragma once

#include <charconv>
#include <cmath>
#include <iomanip>
#include <regex>
#include <sstream>
#include <stddef.h>
#include <stdexcept>
#include <stdint.h>
#include <string>
#include <type_traits>

template<size_t Dim, typename T> struct Point;

#ifdef _MSC_VER
	#define ALWAYSINLINE [[msvc::forceinline]]
#else
	#define ALWAYSINLINE __attribute__((always_inline))
#endif
#if defined(_M_IX86) || defined(_M_X64) || defined(__x86_64__)
	#include <emmintrin.h>
#endif



// A multiplication and an addition or a fused multiply add.
template<typename T> ALWAYSINLINE static inline T MulAdd(T a, T b, T c)
{
#ifdef __AVX2__
	if constexpr (std::is_floating_point<T>::value)
		return std::fma(a, b, c);
	else
#endif
		return a * b + c;
}

// A multiplication and an subtraction or a fused multiply subtract.
template<typename T> ALWAYSINLINE static inline T MulSub(T a, T b, T c)
{
#ifdef __AVX2__
	if constexpr (std::is_floating_point<T>::value)
		return std::fma(a, b, -c);
	else
#endif
		return a * b - c;
}


ALWAYSINLINE static inline int32_t Integerize(double value)
{
// If available we use the processor instruction.
// note that this has slightly different rounding for mid points.
#if defined(_M_IX86) || defined(_M_X64) || defined(__x86_64__)
	return _mm_cvtsd_si32(_mm_max_sd(_mm_set_sd(INT32_MIN), _mm_min_sd(_mm_set_sd(INT32_MAX), _mm_set_sd(value))));
#else
	return (int32_t)std::max((double)INT32_MIN, std::min((double)INT32_MAX, value + (value < 0 ? -0.5 : 0.5)));
#endif
}


namespace Private
{
#ifdef _MSC_VER
	#pragma warning(disable : 4201) // Nonstandard extension used: Nameless struct/union.
// We define the point storage for each dimension for providing "X"-,"Y"-style direct variables in addition to "Array".
#endif
	template<size_t Dim, typename T> struct PointCoordStorage
	{
		T Array[Dim];
		ALWAYSINLINE PointCoordStorage() = default;
	};
	template<typename T> struct PointCoordStorage<2, T>
	{
		union {
			struct
			{
				T X;
				T Y;
			};
			T Array[2];
		};
		ALWAYSINLINE PointCoordStorage() = default;
		ALWAYSINLINE PointCoordStorage(T x, T y)
		    : Array{x, y}
		{}
	};
	template<typename T> struct PointCoordStorage<3, T>
	{
		union {
			struct
			{
				T X;
				T Y;
				T Z;
			};
			T Array[3];
		};
		ALWAYSINLINE PointCoordStorage() = default;
		ALWAYSINLINE PointCoordStorage(T x, T y, T z)
		    : Array{x, y, z}
		{}
	};
	template<typename T> struct PointCoordStorage<4, T>
	{
		union {
			struct
			{
				T X;
				T Y;
				T Z;
				T M;
			};
			T Array[4];
		};
		ALWAYSINLINE PointCoordStorage() = default;
		ALWAYSINLINE PointCoordStorage(T x, T y, T z, T m)
		    : Array{x, y, z, m}
		{}
	};
} // namespace Private

template<size_t Dim, typename T> struct Point : public Private::PointCoordStorage<Dim, T>
{
	using Private::PointCoordStorage<Dim, T>::PointCoordStorage;

	ALWAYSINLINE constexpr Point()
	{
		for (size_t i = 0; i < Dim; ++i)
			this->Array[i] = 0;
	}
	template<typename T2> ALWAYSINLINE constexpr explicit Point(const Point<Dim, T2>& other)
	{
		for (size_t i = 0; i < Dim; ++i)
			this->Array[i] = static_cast<T>(other.Array[i]);
	}
	ALWAYSINLINE constexpr Point(T other)
	{
		for (size_t i = 0; i < Dim; ++i)
			this->Array[i] = other;
	}

	ALWAYSINLINE constexpr T& operator[](size_t index) { return this->Array[index]; }
	ALWAYSINLINE constexpr const T& operator[](size_t index) const { return this->Array[index]; }
	ALWAYSINLINE friend Point operator-(const Point& other)
	{
		Point result;
		for (size_t i = 0; i < Dim; ++i)
			result.Array[i] = -other.Array[i];
		return result;
	}
	ALWAYSINLINE friend Point operator+(const Point& other) { return other; }
	ALWAYSINLINE friend Point operator+(const Point& lhs, const Point& rhs)
	{
		Point result;
		for (size_t i = 0; i < Dim; ++i)
			result.Array[i] = lhs.Array[i] + rhs.Array[i];
		return result;
	}
	ALWAYSINLINE void operator+=(const Point& rhs)
	{
		for (size_t i = 0; i < Dim; ++i)
			this->Array[i] += rhs.Array[i];
	}
	ALWAYSINLINE friend Point operator-(const Point& lhs, const Point& rhs)
	{
		Point result;
		for (size_t i = 0; i < Dim; ++i)
			result.Array[i] = lhs.Array[i] - rhs.Array[i];
		return result;
	}
	ALWAYSINLINE void operator-=(const Point& rhs)
	{
		for (size_t i = 0; i < Dim; ++i)
			this->Array[i] -= rhs.Array[i];
	}
	ALWAYSINLINE friend Point operator*(const Point& lhs, T rhs)
	{
		Point result;
		for (size_t i = 0; i < Dim; ++i)
			result.Array[i] = lhs.Array[i] * rhs;
		return result;
	}
	ALWAYSINLINE friend Point operator*(T lhs, const Point& rhs)
	{
		Point result;
		for (size_t i = 0; i < Dim; ++i)
			result.Array[i] = lhs * rhs[i];
		return result;
	}
	ALWAYSINLINE friend Point operator*(const Point& lhs, const Point& rhs)
	{
		Point result;
		for (size_t i = 0; i < Dim; ++i)
			result.Array[i] = lhs.Array[i] * rhs.Array[i];
		return result;
	}
	ALWAYSINLINE void operator*=(T rhs)
	{
		for (size_t i = 0; i < Dim; ++i)
			this->Array[i] *= rhs;
	}
	ALWAYSINLINE void operator*=(const Point& rhs)
	{
		for (size_t i = 0; i < Dim; ++i)
			this->Array[i] *= rhs.Array[i];
	}
	ALWAYSINLINE friend Point operator/(const Point& lhs, const T& rhs)
	{
		Point result;
		for (size_t i = 0; i < Dim; ++i)
			result.Array[i] = lhs.Array[i] / rhs;
		return result;
	}
	ALWAYSINLINE friend Point operator/(const T& lhs, const Point& rhs)
	{
		Point result;
		for (size_t i = 0; i < Dim; ++i)
			result.Array[i] = lhs / rhs.Array[i];
		return result;
	}
	ALWAYSINLINE friend Point operator/(const Point& lhs, const Point& rhs)
	{
		Point result;
		for (size_t i = 0; i < Dim; ++i)
			result.Array[i] = lhs.Array[i] / rhs.Array[i];
		return result;
	}
	ALWAYSINLINE void operator/=(const Point& rhs)
	{
		for (size_t i = 0; i < Dim; ++i)
			this->Array[i] /= rhs.Array[i];
	}
	ALWAYSINLINE friend bool operator==(const Point& lhs, const Point& rhs)
	{
		for (size_t i = 0; i < Dim; ++i)
			if (!(lhs.Array[i] == rhs.Array[i]))
				return false;
		return true;
	}
	ALWAYSINLINE friend bool operator!=(const Point& lhs, const Point& rhs) { return !(lhs == rhs); }
	ALWAYSINLINE friend Point fabs(const Point& vec)
	{
		using namespace std;
		Point result;
		for (size_t i = 0; i < Dim; ++i)
			result.Array[i] = fabs(vec.Array[i]); // Use fabs not abs because it can use bitwise-and.
		return result;
	}
	ALWAYSINLINE friend Point min(const Point& lhs, const Point& rhs)
	{
		using namespace std;
		Point result;
		for (size_t i = 0; i < Dim; ++i)
			result.Array[i] = min(lhs.Array[i], rhs.Array[i]);
		return result;
	}
	ALWAYSINLINE friend Point max(const Point& lhs, const Point& rhs)
	{
		using namespace std;
		Point result;
		for (size_t i = 0; i < Dim; ++i)
			result.Array[i] = max(lhs.Array[i], rhs.Array[i]);
		return result;
	}
	ALWAYSINLINE friend Point MulAdd(const Point& a, const Point& b, const Point& c)
	{
		Point result;
		for (size_t i = 0; i < Dim; ++i)
			result.Array[i] = MulAdd(a.Array[i], b.Array[i], c.Array[i]);
		return result;
	}
	ALWAYSINLINE friend Point MulSub(const Point& a, const Point& b, const Point& c)
	{
		Point result;
		for (size_t i = 0; i < Dim; ++i)
			result.Array[i] = MulSub(a.Array[i], b.Array[i], c.Array[i]);
		return result;
	}
	ALWAYSINLINE friend Point<Dim, int32_t> Integerize(const Point& vec)
	{
		Point<Dim, int32_t> result;
		for (size_t i = 0; i < Dim; ++i)
			result.Array[i] = Integerize(vec.Array[i]);
		return result;
	}
	ALWAYSINLINE friend T ComponentwiseSum(const Point& vec)
	{
		T result = vec.Array[0];
		for (size_t i = 1; i < Dim; ++i)
			result += vec.Array[i];
		return result;
	}
	ALWAYSINLINE friend T ComponentwiseMax(const Point& vec)
	{
		T result = vec.Array[0];
		for (size_t i = 1; i < Dim; ++i)
			result = std::max(result, vec.Array[i]);
		return result;
	}
	ALWAYSINLINE friend T ComponentwiseMin(const Point& vec)
	{
		T result = vec.Array[0];
		for (size_t i = 1; i < Dim; ++i)
			result = std::min(result, vec.Array[i]);
		return result;
	}
	ALWAYSINLINE friend Point NonFiniteToZero(const Point& vec)
	{
		using namespace std;
		Point result;
		for (size_t i = 0; i < Dim; ++i)
			result.Array[i] = isfinite(vec.Array[i]) ? vec.Array[i] : 0.0;
		return result;
	}

	static inline std::string ToAxisString(size_t index)
	{
		using namespace std;
		if (index >= Dim)
			throw std::invalid_argument("index");
		switch (index)
		{
		case 0:
			return "X";
		case 1:
			return "Y";
		case 2:
			return "Z";
		case 3:
			return "M";
		default:
			return "[" + to_string(index) + "]";
		}
	}
	inline std::string ToString() const
	{
		using namespace std;
		std::stringstream result;
		result << std::setprecision(15);
		for (size_t i = 0; i < Dim; ++i)
		{
			if (i > 0)
				result << " ";
			result << ToAxisString(i) << "=" << this->Array[i];
		}
		return result.str();
	}
	static inline Point FromString(std::string str)
	{
		static const std::regex doubleRgx(R"([-+]?[\d]*\.?[\d]+([Ee][-+]?[\d]+)?)");

		Point result;
		for (size_t i = 0; i < Dim; ++i)
		{
			auto searchResult = std::smatch();
			if (!std::regex_search(str, searchResult, doubleRgx))
				throw std::runtime_error("Unable to parse Point.");
			const std::string& matchStr = searchResult.str();
			const auto error = std::from_chars(matchStr.c_str() + (matchStr[0] == '+'), matchStr.c_str() + matchStr.size(), result.Array[i]);
			if (error.ec != std::errc{})
				throw std::runtime_error("Unable to parse Point.");
			str = searchResult.suffix().str();
		}
		return result;
	}
};

using Point2D = Point<2, double>;
using Point3D = Point<3, double>;
using Point4D = Point<4, double>;
using Point2DInt = Point<2, int32_t>;
using Point3DInt = Point<3, int32_t>;
using Point4DInt = Point<4, int32_t>;

namespace std
{
	template<size_t Dim, typename T> ALWAYSINLINE std::string to_string(const Point<Dim, T>& obj) { return obj.ToString(); }
} // namespace std
